<?php

namespace App\Http\Controllers;

use App\Exports\UsuariosDuplicadosExport;
use App\Imports\ImportUsers;
use App\Models\Charge;
use App\Models\User;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;

class UsuariosController extends Controller
{
    public function index(Request $request)
    {
        $charges = Charge::all();
        $users = User::Usuar($request->UsuarioBus)->orderBy('created_at', 'desc')->paginate(10);

        if ($request->ajax()) {
            return view('partials.tablaUsuarios', compact('users'))->render();
        }

        return view('Usuarios', compact('charges', 'users'));
    }

    // Método para almacenar un nuevo usuario en la base de datos
    public function store(Request $request)
    {
        // Validación de los datos del formulario
        $request->validate([
            'identification' => 'required|integer|unique:users,identification', // Identificación es obligatoria y debe ser un número entero
            'name' => 'required|string', // Nombre es obligatorio y debe ser un texto
            'email' => 'required|email|unique:users,email', // Correo electrónico es obligatorio y debe ser un email válido
            'phone' => 'required|integer', // Teléfono es obligatorio y debe ser un número entero
            'rol' => 'required|string', // Rol es obligatorio y debe ser un texto
            'charge' => 'required|string', // Cargo es obligatorio y debe ser un texto
            'ods' => 'required|string', // ODS es obligatorio y debe ser un texto
            'location' => 'required|string',
            'activo' => 'required|string',
            'vicepresidencia' => 'required|string',
            'gerencia' => 'required|string',
            'emailecp' => 'nullable|email', // Correo electróniico ECP es opcional y debe ser un email válido
        ]);

        // Hasheo de la identificación para utilizarla como contraseña por defecto
        $password = Hash::make($request->identification);
        $state = 'Activo'; // El estado por defecto es 'Activo'

        // Creación de un nuevo registro de usuario en la base de datos
        User::create([
            'identification' => $request->identification,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'ods' => $request->ods,
            'rol' => $request->rol,
            'charge' => $request->charge,
            'password' => $password,
            'state' => $state,
            'location' => $request->location,
            'activo' => $request->activo,
            'vicepresidencia' => $request->vicepresidencia,
            'gerencia' => $request->gerencia,
            'emailecp' => $request->emailecp,

        ]);

        // Retorno de una respuesta en JSON indicando el éxito de la operación
        return response()->json(['success' => 'Usuario creado con éxito']);
    }

    public function edit($id)
    {
        $user = User::find($id);


        return response()->json([
            'charges' => Charge::pluck('charge'), // array de strings
            'name' => $user->name,
            'email' => $user->email,
            'phone' => $user->phone,
            'ods' => $user->ods,
            'rol' => $user->rol,
            'charge_actual' => $user->charge,
            'activo' => $user->activo,
            'vicepresidencia' => $user->vicepresidencia,
            'gerencia' => $user->gerencia,
            'location' => $user->location,
            'emailecp' => $user->emailecp,
        ]);
    }

    public function cambiarEstado(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $estado = $request->input('state');

        $user->state = $estado;
        $user->save();

        return response()->json(['success' => true]);
    }

    public function update(Request $request, $id)
    {
        //
        $request->validate([
            'edit_name' => 'required|string',
            'edit_email' => 'required|email|unique:users,email,' . $id,
            'edit_phone' => 'nullable|integer',
            'edit_ods' => 'required|string',
            'edit_rol' => 'required|string',
            'edit_charge' => 'required|string',
            'edit_activo' => 'required|string',
            'edit_vicepresidencia' => 'required|string',
            'edit_gerencia' => 'required|string',
            'edit_ubicacion' => 'required|string',
            'edit_emailecp' => 'nullable|email',
        ]);

        $user = User::findOrFail($id);

        $user->update([
            'name' => $request->edit_name,
            'email' => $request->edit_email,
            'phone' => $request->edit_phone,
            'ods' => $request->edit_ods,
            'rol' => $request->edit_rol,
            'charge' => $request->edit_charge,
            'activo' => $request->edit_activo,
            'vicepresidencia' => $request->edit_vicepresidencia,
            'gerencia' => $request->edit_gerencia,
            'location' => $request->edit_ubicacion,
            'emailecp' => $request->edit_emailecp,
        ]);

        return response()->json(['success' => 'Registro actualizado']);
    }

    public function import(Request $request)
    {
        $request->validate([
            'archivo_csv' => 'required|file|mimes:csv,txt',
        ]);

        // $path = $request->file('archivo_csv')->getRealPath();
        $data = Excel::toCollection(new ImportUsers, $request->file('archivo_csv'))[0];

        $usuariosDuplicados = [];
        $usuariosNuevos = [];

        foreach ($data as $index => $row) {
            $validator = Validator::make($row->toArray(), [
                'identificacion' => 'required|integer|unique:users,identification',
                'correo' => 'required|email|unique:users,email',
            ]);

            if ($validator->fails()) {
                $usuariosDuplicados[] = $row;
            } else {
                $usuariosNuevos[] = [
                    'identification' => $row['identificacion'],
                    'name' => $row['nombre'],
                    'email' => $row['correo'],
                    'phone' => $row['celular'],
                    'ods' => $row['ods'],
                    'rol' => $row['rol'],
                    'charge' => $row['cargo'],
                    'state' => $row['estado'] ?? 'Activo',
                    'activo' => $row['activo'],
                    'vicepresidencia' => $row['vicepresidencia'],
                    'gerencia' => $row['gerencia'],
                    'location' => $row['localidad'],
                    'password' => Hash::make($row['identificacion']),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
        }

        User::insert($usuariosNuevos);

        if (count($usuariosDuplicados) > 0) {
            $filename = 'usuarios_duplicados_' . now()->format('Ymd_His') . '.xlsx';

            session()->put('duplicados_export', new UsuariosDuplicadosExport(collect($usuariosDuplicados)));
            session()->put('duplicados_filename', $filename);

            return response()->json([
                'warning' => 'Algunos usuarios ya están registrados.',
                'download' => route('usuarios.duplicados.descargar')
            ]);
        }

        return response()->json(['success' => 'Usuarios importados correctamente']);
    }
}
