<?php

namespace App\Http\Controllers;

use App\Models\Category; // Modelo que gestiona las categorías de productos
use App\Models\Report; // Modelo que gestiona los reportes de productos
use App\Models\reportTimeModel; // Modelo que gestiona los registros de tiempo de los reportes
use App\Models\Service; // Modelo que gestiona los servicios
use App\Models\Approver; // modelo que gestiona a los aprobadores
use App\Models\Codigo_sig;
use App\Models\New_fn;
use App\Models\User;
use Carbon\Carbon; // Biblioteca para manejo de fechas y tiempos
use Illuminate\Http\Request; // Manejo de solicitudes HTTP
use Illuminate\Support\Facades\Auth; // Proporciona funcionalidad de autenticación

class ReporteProductoController extends Controller
{
    // Muestra la vista principal para la gestión de reportes de productos
    public function index()
    {
        // Obtiene la identificación del usuario autenticado
        $UserId = Auth::user()->identification;

        $usuario = User::where('identification', $UserId)->first();

        $productos = Category::where('sector', $usuario->charge)->select('product')
            ->orderBy('product', 'asc')
            ->get();

        //obtenemos los fn que pertenecen a la identificación de la persona que esta autenticada.
        $fnasignados = New_fn::select('fn')
            ->where('identification', $UserId)
            ->get();

        // Obtiene todos los servicios disponibles
        $servicios = Service::all();

        // Obtiene los reportes pendientes del usuario autenticado
        $reportes = Report::where('professional', $UserId)
            ->orderBy('date')
            ->where('state', 'pendiente')
            ->get();

        // Obtenemos el sector a lo que pertenece cada producto, asegurandonos de que no se repitan
        $sectores = Category::select('sector')->distinct()->get();

        // Obtiene las categorías de productos, asegurándose de que no se repitan
        $categorias = Category::select('product')->distinct()->get();


        $talentos = User::where('charge', 'talento humano')
            ->get();
        //
        $primero = $talentos->first();
        $aprobadores = Approver::all();

        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;
        $timeRep = reportTimeModel::where('identification', $UserId)
            ->whereMonth('date', $currentMonth)
            ->whereYear('date', $currentYear)
            ->orderBy('date')
            ->get();

        // Crear un array de eventos con múltiples descripciones por día
        $events = [];
        foreach ($timeRep as $report) {
            $date = \Carbon\Carbon::parse($report->date)->format('Y-m-d');
            $events[$date][] = [
                'description' => $report->description
            ];
        }

        // Generar el calendario para el mes actual
        $firstDay = Carbon::create($currentYear, $currentMonth, 1);
        $lastDay = $firstDay->copy()->endOfMonth();
        $calendar = [];
        $currentDate = $firstDay->copy()->startOfWeek();

        while ($currentDate->lte($lastDay)) {
            $formattedDate = $currentDate->format('Y-m-d');
            $calendar[$formattedDate] = $events[$formattedDate] ?? [];
            $currentDate->addDay();
        }

        // Retorna la vista con los datos necesarios
        return view('reporteProducto', compact('calendar', 'currentMonth', 'usuario', 'timeRep', 'productos', 'servicios', 'sectores', 'categorias', 'reportes', 'aprobadores', 'talentos', 'fnasignados', 'primero'));
    }

    // Obtiene el producto de un sector específico
    public function getProductosBySector(Request $request)
    {
        // Busca los productos asociados al sector enviado en la solicitud
        $productos = Category::where('sector', $request->sector)->select('product')->get(); //se le quito el ->distinct() que va antes del ->get(), para que pueda imprimir todos los productos de ese perfil seleccionado.

        // Retorna los productos o un mensaje de "No encontrado" si no existen
        return response()->json($productos);
    }

    // Obtiene la categoría de un producto específico
    public function getCategoria(Request $request)
    {
        // Busca la categoría asociada al producto enviado en la solicitud
        $categoria = Category::where('product', $request->producto)->select('category')->first();

        // Retorna la categoría o un mensaje de "No encontrado" si no existe
        return response()->json(['category' => $categoria ? $categoria->category : 'No encontrado']);
    }

    public function getServicio(Request $request)
    {
        //buscamos el servicio que le pertenece a la fn seleccionada
        $servicio = New_fn::where('fn', $request->fn)->select('service')->first();
        $codigos = Codigo_sig::where('fn', $request->fn)->select('codigo_sig')->get();

        $codigoArray = $codigos->pluck('codigo_sig');

        return response()->json([
            'service' => $servicio ? $servicio->service : null,
            'codigo_sig' => $codigoArray
        ]);
    }


    // Almacena un nuevo reporte de producto en la base de datos
    public function store(Request $request)
    {
        // Estado inicial del reporte
        $estado = 'pendiente';
        $ubicacionSelect = $request->input('ubicacion');
        $activoSelect = $request->input('activo');

        if ($ubicacionSelect === 'mantener') {
            $Userid = Auth::user()->identification;
            $ubicacionSelect = User::where('identification', $Userid)->value('location');
        }
        if ($activoSelect === 'mantener') {
            $Userid = Auth::user()->identification;
            $activoSelect = User::where('identification', $Userid)->value('activo');
        }

        // Valida los datos del formulario de creación de report
        $request->validate([
            'producto' => 'nullable|string',
            'descripcion' => 'nullable|string',
            'categoria' => 'required|string',
            'fecha' => 'nullable|date',
            'fn' => 'nullable|string',
            'codigo' => 'nullable|string',
            'servicio' => 'nullable|string',
            'location' => 'nullable|string',
            'activo' => 'required|string',

            // PMS
            'pms' => 'nullable|string',
            'tipo_hito' => 'nullable|string',
            'hito_clave_solicitud' => 'nullable|string',
            'hito_clave_solicitud_nopplaneado' => 'nullable|string',
            'valoracion' => 'nullable|string',
            'justificacion' => 'nullable|string',
            'funcionario' => 'nullable|string',
            'horas' => 'nullable|string',
            'ruta_url' => 'nullable|string',
            'ruta_url_np' => 'nullable|string',
            'servicios' => 'nullable|array',
            'servicios_no_planeado' => 'nullable|array',
        ]);

        // dd($request);

        // Concatenar arrays si se usan checkbox múltiples
        $servicios = $request->input('servicios') ? implode(', ', $request->input('servicios')) : null;
        $serviciosNoPlaneado = $request->input('servicios_no_planeado') ? implode('/ ', $request->input('servicios_no_planeado')) : null;

        // Convierte fn y codigo a mayúsculas
        $fnUpper = strtoupper($request->fn);

        // Crea un nuevo reporte con los datos validados
        $newReport = Report::create([
            'charge' => Auth::user()->charge, // Cargo del usuario autenticado
            'professional' => Auth::user()->identification, // Identificación del profesional
            'product' => $request->producto,
            'description' => $request->descripcion,
            'category' => $request->categoria,
            'date' => $request->fecha,
            'fn' => $fnUpper,
            'code_sig' => $request->codigo,
            'service' => $request->servicio,
            //'location' => $request->ubicacion,
            'location' => $ubicacionSelect,
            'activo' => $activoSelect,
            'state' => $estado,

            // PMS
            'id_gestion_predial_del_pms' => $request->pms ?? null,
            'tipo_de_hito_clave_de_control' => $request->tipo_hito ?? null,
            'hito_clave_de_control_solicitud_servicio' => $request->hito_clave_solicitud ?? null,
            'servicios_participan_elaboracion_del_hito' => $servicios ?? $serviciosNoPlaneado,
            'ruta_url' => $request->ruta_url ?? $request->ruta_url_np ?? null,

            'hito_clave_de_control_no_planeado_directriz' => $request->hito_clave_solicitud_nopplaneado ?? null,
            'valoracion_de_resultado' => $request->valoracion ?? null,
            'justificacion_del_impacto_alto' => $request->justificacion ?? null,
            'funcionario_solicitante_no_planeado' => $request->funcionario ?? null,
            'horas_acumuladas_al_cumplimiento_hito' => $request->horas ?? null,
        ]);

        $reporteId = $newReport->id;

        // Crea un nuevo registro en la tabla de tiempos de reportes
        $post = new reportTimeModel;
        $post->identification = Auth::user()->identification;
        $post->date = $request->fecha;
        $post->location = $ubicacionSelect;
        $post->description = $request->descripcion;
        $post->id_report = $reporteId;
        $post->save();


        // Retorna un mensaje de éxito
        return response()->json(['success' => 'Tarea Registrada']);
    }

    public function storeTime(Request $request)
    {
        // Estado inicial del reporte
        $ubicacionSelect = $request->input('ubicacion');

        if ($ubicacionSelect === 'mantener') {
            $Userid = Auth::user()->identification;
            $ubicacionSelect = User::where('identification', $Userid)->value('location');
        }

        // Valida los datos del formulario
        $request->validate([
            'descripcion' => 'nullable|string',
            'fecha' => 'nullable|date',
            'location' => 'nullable|string',
        ]);

        // Crea un nuevo registro en la tabla de tiempos de reportes
        $post = new reportTimeModel;
        $post->identification = Auth::user()->identification;
        $post->date = $request->fecha;
        $post->location = $ubicacionSelect;
        $post->description = $request->descripcion;
        $post->save();

        // Retorna un mensaje de éxito
        return response()->json(['success' => 'Tiempo Registrado']);
    }


    // Edita un reporte existente
    public function edit($id)
    {
        // Busca el reporte por su ID y lo retorna como JSON
        $report = Report::findOrFail($id);
        return response()->json($report);
    }

    // Actualiza un reporte existente
    public function update(Request $request, $id)
    {
        // Busca el reporte por su ID y lo actualiza
        $report = Report::findOrFail($id);

        // Estado final del reporte
        $estado = 'finalizado';
        //$desc = $report->description;


        $request->validate([
            'datofecha' => 'required|date',
            'link' => 'required|string',
        ]);

        $report->update([
            'date' => $request->datofecha, // Actualiza la fecha del reporte a la fecha actual
            'link' => $request->link, // Agrega el enlace del reporte
            'ruta_url' => $request->link, // Agrega el enlace del reporte
            'state' => $estado, // Cambia el estado a finalizado
        ]);

        /*  if ($request->has('newTime')) {
            $update = new reportTimeModel;
            $update->identification = Auth::user()->identification;
            $update->date = $request->datofecha;
            $update->location = $report->location;
            $update->description = $report->description;
            $update->id_report = $report->id;
            $update->save();
        } */

        // Retorna un mensaje de éxito
        return response()->json(['success' => 'Producto finalizado']);
    }


    // Extiende la fecha de un reporte pendiente
    public function extend(Request $request, $id)
    {
        // Validar que el reporte esté en estado 'pendiente'
        $report = Report::findOrFail($id);
        $fechaNova = $request->fechaNov;

        if ($report->state !== 'pendiente') {
            return response()->json(['Error' => 'Solo se pueden extender tareas pendientes'], 400);
        }
        $request->validate([
            'fechaNov' => 'required|date', // Cambiado de 'email' a 'date'
            'descripNov' => 'required|string',
        ]);

        if ($report->date === $fechaNova) {
            return response()->json(['Error' => 'No puede extender con la misma fecha'], 400);
        };

        reportTimeModel::create([
            'identification' => $report->professional,
            'date' => $request->fechaNov,
            'location' => $report->location,
            'description' => $request->descripNov,
            'id_report' => $report->id,
        ]);

        $report->update([
            'date' => $request->fechaNov, // Actualizar solo la fecha a la fecha actual
        ]);

        return response()->json(['success' => 'Tarea extendida']);
    }

    public function terminar(Request $request, $id)
    {
        $report = Report::findOrFail($id);

        $estado = 'finalizado';
        $report->update([
            'state' => $estado, // Cambia el estado a finalizado
        ]);
    }

    public function eliminar(Request $request, $id)
    {
        $report = Report::findOrFail($id);
        if ($report) {
            reportTimeModel::where('id_report', $id)->delete();
            $report->delete();

            // Retorna una respuesta en JSON indicando el éxito de la operación
            return response()->json(['success' => 'Reporte eliminado con éxito']);
        }

        // Si el usuario no se encuentra, retorna un error 404
        return response()->json(['error' => 'Usuario no encontrado'], 404);
    }

    public function actualizar(Request $request, $id)
    {
        $request->validate([
            'fecha2' => 'required|date', // Cambiado de 'email' a 'date'
            'fn2' => 'required|string', // Asegúrate de que el nombre del campo coincide
            'codeSIG' => 'required|string',
            'descripcion2' => 'required|string',
        ]);

        $report = Report::findOrFail($id);

        $report->update([
            'date' => $request->fecha2,
            'fn' => $request->fn2, // Asegúrate de que el nombre del campo coincide
            'code_sig' => $request->codeSIG,
            'description' => $request->descripcion2,
        ]);

        //$reportTime = ReportTimeModel::where('id_report', $report->id)->first();
        $reportTime = ReportTimeModel::where('id_report', $report->id)->first();


        $reportTime->update([
            'date' => $request->fecha2,
            'description' => $request->descripcion2,
        ]);


        return response()->json(['success' => 'Actualizacion de datos registrada exitosamente']);
    }
}
